package net.kldp.beat.action;

import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.kldp.beat.dispatcher.Dispatch;
import net.kldp.beat.dispatcher.ViewDispatcher;
import net.kldp.beat.map.ApplicationMap;
import net.kldp.beat.map.CookieMap;
import net.kldp.beat.map.HeaderMap;
import net.kldp.beat.map.ParameterMap;
import net.kldp.beat.map.RequestMap;
import net.kldp.beat.map.SessionMap;

/**
 * 전체 어플리케이션에서 공유하는 객체를 감싸는 클래스입니다. request, response, dispatcher와 기타 aware류의
 * 맵들을 생성해서 보관 유지합니다. 이들 객체는 시스템 인터셉터가 액션 객체에 주입하기 위해 사용됩니다.
 */
public class ActionContext {

	private Map<String, Object> parameterMap;
	private Map<String, Object> requestMap;
	private Map<String, Object> sessionMap;
	private Map<String, Object> applicationMap;
	private HttpServletRequest request;
	private HttpServletResponse response;
	private ServletContext context;
	private Dispatch dispatcher;
	private CookieMap cookieMap;
	private HeaderMap headerMap;

	public ActionContext(HttpServletRequest request, HttpServletResponse response, ServletContext context)
			throws ServletException {
		this.request = request;
		this.response = response;
		this.context = context;
		dispatcher = ViewDispatcher.create(context, request, response);
	}

	/**
	 * Request scope의 맵을 리턴합니다.
	 * 
	 * @return
	 */
	public Map<String, Object> getRequestMap() {
		if (requestMap == null) {
			requestMap = new RequestMap(request);
		}
		return requestMap;
	}

	/**
	 * Application scope의 맵을 리턴합니다.
	 * 
	 * @return
	 */
	public Map<String, Object> getApplicationMap() {
		if (applicationMap == null) {
			applicationMap = new ApplicationMap(context);
		}
		return applicationMap;
	}

	/**
	 * Session의 맵을 리턴합니다.
	 * 
	 * @return
	 */
	public Map<String, Object> getSessionMap() {
		if (sessionMap == null) {
			sessionMap = new SessionMap(request.getSession());
		}
		return sessionMap;
	}

	/**
	 * 요청 파라미터의 맵을 리턴합니다.
	 * 
	 * @return
	 */
	public Map<String, Object> getParameterMap() {
		if (parameterMap == null) {
			parameterMap = new ParameterMap(request);
		}
		return parameterMap;
	}

	/**
	 * ServletContext 객체를 리턴합니다.
	 * 
	 * @return
	 */
	public ServletContext getServletContext() {
		return context;
	}

	/**
	 * ServletRequest 객체를 리턴합니다.
	 * 
	 * @return
	 */
	public HttpServletRequest getServletRequest() {
		return request;
	}

	/**
	 * ServletResponse 객체를 리턴합니다.
	 * 
	 * @return
	 */
	public HttpServletResponse getServletResponse() {
		return response;
	}

	public Dispatch getDispatcher() {
		return dispatcher;
	}

	/**
	 * 요청 파라미터 맵을 설정합니다. 요청 방식이 Multipart/Form-data형식일때 사용됩니다.
	 * 
	 * @param parameterMap
	 */
	public void setParameterMap(Map<String, Object> parameterMap) {
		this.parameterMap = parameterMap;
	}

	public Map<String, Cookie> getCookieMap() {
		if (cookieMap == null)
			cookieMap = new CookieMap(request.getCookies(), response);
		return cookieMap;
	}

	public Map<String, String> getHeaderMap() {
		if (headerMap == null)
			headerMap = new HeaderMap(request, response);
		return headerMap;
	}
}