package net.kldp.beat.action;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.kldp.beat.action.pool.AnnotationPool;
import net.kldp.beat.annotation.Beat;

/**
 * 클래스에 정의된 모든 어노테이션을 검색합니다.
 * 
 */
public class AnnotationFinder {
	private Map<Class<? extends Annotation>, Annotation> annotations = new HashMap<Class<? extends Annotation>, Annotation>();
	private Class<?> actionClass;

	public AnnotationFinder(Class<?> actionClass) {
		this.actionClass = actionClass;
	}

	/**
	 * 모든 정의된 어노테이션들을 검색합니다.
	 * 
	 * @param actionClass
	 * @return
	 */
	public void find() {
		annotations = AnnotationPool.get(actionClass);
		if (annotations == null)
			findAll();
	}

	private void findAll() {
		// 액션 클래스 및 상속 클래스의 어노테이션들
		annotations = findAnnotation(actionClass);

		// 액션 클래스가 구현하는 인터페이스의 어노테이션들
		for (Class<?> clazz : actionClass.getInterfaces()) {
			putAnnotation(findAnnotation(clazz));
		}

		// 액션 클래스의 부모 클래스의 인터페이스의 어노테이션들
		Class<?> superClass = actionClass.getSuperclass();
		while (!superClass.equals(Object.class)) {
			for (Class<?> clazz : superClass.getInterfaces()) {
				putAnnotation(findAnnotation(clazz));
			}
			superClass = superClass.getSuperclass();
		}
		AnnotationPool.put(actionClass, annotations);
	}

	/**
	 * 중복되지 않는 어노테이션을 삽입합니다.
	 * 
	 * @param map
	 */
	private void putAnnotation(Map<Class<? extends Annotation>, Annotation> map) {
		for (Class<? extends Annotation> clazz : map.keySet()) {
			if (!annotations.containsKey(clazz)) {
				annotations.put(clazz, map.get(clazz));
			}
		}
	}

	/**
	 * 현재 클래스의 어노테이션을 검색합니다.
	 * 
	 * @param clazz
	 * @return
	 */
	private Map<Class<? extends Annotation>, Annotation> findAnnotation(
			Class<?> clazz) {
		Map<Class<? extends Annotation>, Annotation> map = new HashMap<Class<? extends Annotation>, Annotation>();
		for (Annotation annotation : clazz.getAnnotations()) {
			if (isBeat(annotation)) {
				map.put(annotation.getClass(), annotation);
			}
		}
		return map;
	}

	/**
	 * 검색된 모든 어노테이션을 리턴합니다.
	 * 
	 * @return
	 */
	public List<Annotation> getAnnotations() {
		return new ArrayList<Annotation>(annotations.values());
	}

	/**
	 * 어노테이션이 Beat어노테이션을 정의했는지 검사합니다.
	 * 
	 * @param annotation
	 * @return
	 */
	public static boolean isBeat(Annotation annotation) {
		for (Annotation anno : annotation.annotationType().getAnnotations()) {
			if (anno instanceof Beat)
				return true;
		}
		return false;
	}
}