/*
    sayclubaccount.cpp - Kopete Sayclub Protocol

    Copyright (c) 2006      by Park J. K.		 <nemesis@planetmono.org>
    Kopete    (c) 2002-2003 by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU General Public                   *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#include "sayclubaccount.h"

#include <qstring.h>
#include <qcstring.h>
#include <qptrlist.h>
#include <qtextcodec.h>

#include <kaction.h>
#include <kdebug.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kinputdialog.h>

#include <kopeteglobal.h>
#include <kopeteuiglobal.h>
#include <kopetemetacontact.h>
#include <kopetecontactlist.h>
#include <kopetegroup.h>
#include <kopetecontactproperty.h>
#include <kopetemessage.h>
#include <kopetechatsession.h>

#include "sayclubcontact.h"
#include "sayclubserver.h"
#include "sayclubprotocol.h"
#include "gaea_phpobject.h"

SayclubAccount::SayclubAccount( SayclubProtocol *parent, const QString& accountID, const char *name )
	: Kopete::PasswordedAccount( parent, accountID , 0, name )
{
	srand( (unsigned int)time( (time_t *)NULL ) );
	m_connectstatus = SayclubProtocol::protocol()->sayclubOnline;
	setMyself( new SayclubContact( this, accountId(), Kopete::ContactList::self()->myself() ) );
	myself()->setOnlineStatus( SayclubProtocol::protocol()->sayclubOffline );
	m_server = NULL;
	
	// Initialize parameters
	m_connected = false;
	m_away = false;
	m_oldStatus = SayclubProtocol::protocol()->sayclubOnline;
	m_keyno = rand()%899999+100000;
	m_key = m_id+QString::number(m_keyno);
	m_uuid = QString( SayclubUtilities::pseudoUUID() );
	SayclubUtilities::ifInterf *ifinterf = new SayclubUtilities::ifInterf;
	if( !ifinterf->isSane() ) {
		kdDebug( 14210 ) << "ifInterf is invalid!" << endl;
		emit errorMessage( ErrorNormal, QString::fromUtf8( "로컬 IP 주소를 알아낼 수 없습니다." ) );
		m_ipaddr = QString( "" );
		m_macaddr = QString( "" );
	}
	else {
		m_ipaddr = ifinterf->getIPAddr();
		m_macaddr = ifinterf->getMacAddr();
	}
	delete ifinterf;
	
	// Attempt to load CP949 Codec, if cp949 has not found, load euc-kr codec instead.
	m_krcodec = QTextCodec::codecForName( "CP949" );
	if(m_krcodec == NULL)
		m_krcodec = QTextCodec::codecForName( "euc-kr" );
	
	QPtrList<Kopete::Group> groupList = Kopete::ContactList::self()->groups();
	for( Kopete::Group *g = groupList.first(); g; g = groupList.next() ) {
		QString groupSn = g->pluginData( protocol(), accountId() + " groupsn" );
		m_groupList.replace( groupSn, g );
	}
	
	// Insert submenu
	m_changeDNAction = new KAction( QString::fromUtf8( "대화명 번경(&C)" ), QString::null, 0, this, SLOT( slotChangeDisplayName() ), this, "renameAction" );
	
	m_id = accountID;
	QObject::connect( this, SIGNAL( accountClosed( Kopete::Account::DisconnectReason ) ), this, SLOT( disconnected( Kopete::Account::DisconnectReason ) ) );
	QObject::connect( this, SIGNAL( errorMessage( int, const QString & ) ), this, SLOT( slotErrorMessageReceived( int, const QString & ) ) );
	QObject::connect( this, SIGNAL( statusChanged( const Kopete::OnlineStatus & ) ), this, SLOT( slotChangeStatus( const Kopete::OnlineStatus & ) ) );
}

SayclubAccount::~SayclubAccount()
{
	QObject::disconnect( this, 0, this, 0 );
	
	if( m_server ) {
		delete m_server;
		m_server = NULL;
	}
}

KActionMenu* SayclubAccount::actionMenu()
{
	KActionMenu *actionMenu = Kopete::Account::actionMenu();
	
	if( m_connected ) {
		m_changeDNAction->setEnabled( true );
	}
	else {
		m_changeDNAction->setEnabled( false );
	}
	
	actionMenu->popupMenu()->insertSeparator();
	actionMenu->insert( m_changeDNAction );
	
	return actionMenu;
}

bool SayclubAccount::createContact(const QString& contactId, Kopete::MetaContact* parentContact)
{
	SayclubContact* newContact = new SayclubContact( this, contactId, parentContact );
	newContact->setProperty( Kopete::Global::Properties::self()->nickName(), parentContact->displayName() );
	
	return newContact != NULL;
}

// Change online status & away message
void SayclubAccount::setOnlineStatus(const Kopete::OnlineStatus& status, const QString &reason )
{
	if( m_server && !m_server->isValid() ) {
		delete m_server;
		m_server = NULL;
		m_connected = false;
	}
	
	// Case #1: If you are online and selected offline status, disconnect with current server
	if( status.status() == Kopete::OnlineStatus::Offline && myself()->onlineStatus() != SayclubProtocol::protocol()->sayclubOffline ) {
		kdDebug( 14210 ) << k_funcinfo << "disconnecting" << endl;
		disconnect();
		m_connected = false;
		emit accountClosed( Kopete::Account::Manual );
	}
	// Case #2: IF you are online, change online status and away message
	else if( m_server && m_connected ) {
		kdDebug( 14210 ) << k_funcinfo << "setting online status" << endl;
		{
			PHPObject o = PHPMapObject();
			PHPMapObject &m = o.asMap();
			m["method"] = string( "request_updateuserinfo" );
			m["onlinestatus"] = statusToId( (Kopete::OnlineStatus &)status );
			m_server->sendCmd( "request_updateuserinfo", o );
			myself()->setOnlineStatus( status );
		}
		{
			QVariant personalMsg = myself()->property( SayclubProtocol::protocol()->propPersonalMessage ).value();
			QCString encoded;
			if( reason.isNull() )
				encoded = "";
			else {
				encoded = m_krcodec->fromUnicode( reason );
				encoded.truncate( 40 );
			}
			PHPObject o = PHPMapObject();
			PHPMapObject &m = o.asMap();
			m["method"] = string( "request_updateuserinfo" );
			m["MEMBER_TACHY_OTHER_COMMENT"] = string( encoded );
			m_server->sendCmd( "request_updateuserinfo", o );
			myself()->setProperty( SayclubProtocol::protocol()->propPersonalMessage, m_krcodec->toUnicode( encoded ) );
		}
	}
	// Case #3: If you are offline, connect to the server
	else if( !m_server && !m_connected ) {
		kdDebug( 14210 ) << k_funcinfo << "connecting" << endl;
		m_connectstatus = status;
		connect();
	}
}

void SayclubAccount::connectWithPassword( const QString &password )
{
	kdDebug ( 14210 ) << k_funcinfo << endl;
	
	if( m_connected )
		return;
	
	if( m_server ) {
		QObject::disconnect( m_server, 0, this, 0 );
		QObject::disconnect( this, 0, this, 0 );
		delete m_server;
		m_server = NULL;
	}
	myself()->setOnlineStatus( SayclubProtocol::protocol()->statConnecting );
	
	m_server = new SayclubServer( this );
	QObject::connect( m_server, SIGNAL( errorMessage( int, const QString & ) ), this, SLOT( slotErrorMessageReceived( int, const QString & ) ) );
	QObject::connect( m_server, SIGNAL( receivedHello() ), this, SLOT( slotNegotiate() ) );
	QObject::connect( m_server, SIGNAL( endNegotiation() ), this, SLOT( slotAuthenticate() ) );
	QObject::connect( this, SIGNAL( closeSocket() ), m_server, SLOT( slotSocketClose() ) );
	
	m_password = password;
	m_server->connect( SERVERADDR, 25 );
}

void SayclubAccount::disconnect()
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	myself()->setOnlineStatus( SayclubProtocol::protocol()->sayclubOffline );
	if( m_server ) {
		m_server->disconnect();
		delete m_server;
		m_server = NULL;
	}
}

SayclubServer * SayclubAccount::server()
{
	return m_server;
}

// Send essential information to the server
void SayclubAccount::slotNegotiate()
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	m_server->write( QString( "IRCR_KEY %1\nIRCR_NICK %2\nPEER_SWITCH IM_SESSION" ).arg( m_key ).arg( m_id ).latin1() );
}

// Error Message Handling
void SayclubAccount::slotErrorMessageReceived( int type, const QString &msg )
{
	KMessageBox::DialogType msgBoxType;
	QString caption = QString::fromUtf8("세이클럽 플러그인");
	
	switch(type) {
		case SayclubServer::ErrorNormal:
			msgBoxType = KMessageBox::Error;
			break;
		case SayclubServer::ErrorSorry:
			msgBoxType = KMessageBox::Sorry;
			break;
		case SayclubServer::ErrorInformation:
			msgBoxType = KMessageBox::Information;
			break;
		case SayclubServer::ErrorInternal:
			msgBoxType = KMessageBox::Error;
			caption = QString::fromUtf8("세이클럽 플러그인 내부 오류");
			break;
		default:
			msgBoxType = KMessageBox::Error;
			break;
	}
	
	KMessageBox::queuedMessageBox( Kopete::UI::Global::mainWidget(), msgBoxType, msg, caption );
}

// Change my status
void SayclubAccount::slotChangeStatus( const Kopete::OnlineStatus &o )
{
	myself()->setOnlineStatus( o );
}

// Change OnlineStatus to all contacts (includes me)
void SayclubAccount::slotChangeAllStatus( const Kopete::OnlineStatus &o )
{
	myself()->setOnlineStatus( o );
	setAllContactsStatus( o );
}

// Change loginstatus of specified contact
void SayclubAccount::slotChangeContactLoginStatus( PHPMapObject &m )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	SayclubContact *c;
	
	if( m["friendid"].isNull() ) // On occasion, server does not specify "friendid" variable. Then, we must find the contact by msrl.
		c = getContactByMsrl( std2qstr( m["friendmsrl"].asString() ) );
	else
		c = static_cast<SayclubContact *>( contacts()[m["friendid"].asString()] );
	
	if( !c ) {
		kdDebug( 14210 ) << k_funcinfo << QString::fromUtf8( "서버가 등록되지 않은 ID에 대한 로그인 상태를 보내왔습니다.\n(%1)" ).arg( m["friendid"].isNull() ? QString( "msrl : %1" ).arg( std2qstr( m["friendmsrl"].asString() ) ):QString( "ID : %1" ).arg( std2qstr( m["friendid"].asString() ) ) ) << endl;
		return;
	}
	
	if( m["action"].asString() == "login" )
		c->setOnlineStatus( SayclubProtocol::protocol()->sayclubOnline );
	else if( m["action"].asString() == "logout" )
		c->setOnlineStatus( SayclubProtocol::protocol()->sayclubOffline );
	else
		kdDebug( 14210 ) << k_funcinfo << "Unknown action: " << m["action"].asString() << endl;
}

void SayclubAccount::slotChangeContactStatus( PHPMapObject &m )
{
	kdDebug ( 14210 ) << k_funcinfo << endl;
	
	SayclubContact *c;
	
	if( m["friendid"].isNull() ) // On occasion, server does not specify "friendid" variable. Then, we must find the contact by msrl value.
		c = getContactByMsrl( std2qstr( m["friendmsrl"].asString() ) );
	else if( std2qstr( m["friendid"].asString() ) == accountId() )
		return;
	else
		c = static_cast<SayclubContact *>( contacts()[m["friendid"].asString()] );
	
	if( c == myself() )
		return;
	
	if( !c ) {
		kdDebug( 14210 ) << k_funcinfo << QString::fromUtf8( "서버가 등록되지 않은 ID에 대한 로그인 상태를 보내왔습니다.\n(%1)" ).arg( m["friendid"].isNull() ? QString( "msrl : %1" ).arg( std2qstr( m["friendmsrl"].asString() ) ):QString( "ID : %1" ).arg( std2qstr( m["friendid"].asString() ) ) ) << endl;
		return;
	}
	
	if( !m["onlinestatus"].isNull() ) {
		string os = m["onlinestatus"].asString();
		if( os == "2" || os == "1" ) // Online
			c->setOnlineStatus( SayclubProtocol::protocol()->sayclubOnline );
		else if( os == "3" || os == "5" ) // Away
			c->setOnlineStatus( SayclubProtocol::protocol()->sayclubAway );
		else if( os == "4" ) // Busy
			c->setOnlineStatus( SayclubProtocol::protocol()->sayclubBusy );
		else // Unknown
			c->setOnlineStatus( SayclubProtocol::protocol()->sayclubOffline );
	}
	QString username = std2qstrenc( m["friendname"].asString(), m_krcodec );
	QString mynoti = std2qstrenc( m["mynoti"].asString(), m_krcodec );
	
	c->setProperty( Kopete::Global::Properties::self()->nickName(), username );
	c->setProperty( SayclubProtocol::protocol()->propPersonalMessage, mynoti );
}

// Incoming message
void SayclubAccount::slotIncomingMemo( PHPMapObject &m )
{
	SayclubContact *from = getContactByMsrl( std2qstr( m["sendermsrl"].asString() ) );
	
	if( !from ) return;
	Kopete::Message msg( from, from->manager()->members(), std2qstrenc( m["content"].asString(), m_krcodec ), Kopete::Message::Inbound );
	from->manager( Kopete::Contact::CanCreate )->appendMessage( msg );
}

// Authenticate
void SayclubAccount::slotAuthenticate()
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	m_server->write( QString( "IRCR_OPTIONSET APPSVR_REQURL_SERIAL :/im/phpserver/imserver2.nwz\n" ).latin1() );
	
	PHPObject t = PHPMapObject();
	PHPMapObject &o = t.asMap();
	o["myip"] = qstr2std( m_server->m_p_myip );
	o["method"] = string( "request_login" );
	o["force"] = string( "0" );
	o["clientip"] = qstr2std( m_ipaddr );
	o["key"] = qstr2std( m_key );
	o["usrid"] = qstr2std( m_id );
	o["uuid"] = qstr2std( m_uuid );
	o["myport"] = qstr2std( QString::number( m_server->m_p_myport ) );
	o["checkpw"] = string( "0" );
	o["macaddress"] = qstr2std( m_macaddr );
	o["passwd"] = qstr2std( m_password );
	
	m_server->sendCmd( "request_login", t, this, SLOT( slotParseAuthenticationResult( int ) ) );
}

void SayclubAccount::slotParseAuthenticationResult( int rq )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	if( getErrorCode( o ) != -1 ) {
		emit errorMessage( SayclubServer::ErrorNormal, std2qstrenc( o.asMap()["errstr"].asString(), m_krcodec ) );
		emit closeSocket();
		A_REMOVEREQ( rq );
		return;
	}
	
	A_REMOVEREQ( rq );
	
	PHPObject t = PHPMapObject();
	t.asMap()["method"] = string( "request_startinfo" );
	m_server->sendCmd( "request_startinfo", t, this, SLOT( slotFetchBuddyList(int) ) );
}

void SayclubAccount::slotFetchBuddyList( int rq )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	// Now online!
	{
		PHPObject o = PHPMapObject();
		PHPMapObject &m = o.asMap();
		m["method"] = string( "request_updateuserinfo" );
		m["onlinestatus"] = statusToId( m_connectstatus );
		m_server->sendCmd( "request_updateuserinfo", o );
		emit statusChanged( m_connectstatus );
		m_server->setOnlineStatus( SayclubServer::Connected );
	}
	m_connected = true;
	
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	// Group first
	{
		PHPArrayObject &groups = o.asMap()["GROUPLIST"].asArray();
		QPtrList<Kopete::Group> groupList = Kopete::ContactList::self()->groups();
		Kopete::Group *fallBack;
		
		// Cycle grouplist
		for( PHPArrayObject::iterator it = groups.begin(); it != groups.end(); ++it ) {
			PHPMapObject &m = it->asMap();
			QString groupSn = QString::number( m["groupsn"].asInt() );
			QString groupName = std2qstrenc( m["groupname"].asString(), m_krcodec );
			fallBack = NULL;
			
			// First Attempt
			for( Kopete::Group *g = groupList.first(); g; g = groupList.next() )
				if( g && g->pluginData( protocol(), accountId()+" groupsn" ) == groupSn ) {
					fallBack = g;
					break;
				}
			
			// Second Attempt
			if( !fallBack )
				for( Kopete::Group *g = groupList.first(); g; g = groupList.next() )
					if( g->displayName() == groupName && g->pluginData( protocol(), accountId()+" groupsn" ).isEmpty() ) {
						fallBack = g;
						break;
					}
			
			// If prior attempts failed, Create new one.
			if( !fallBack ) {
				fallBack = new Kopete::Group( groupName );
				Kopete::ContactList::self()->addGroup( fallBack );
			}
			
			// Update Group Information
			fallBack->setPluginData( protocol(), accountId()+" groupsn", groupSn );
			fallBack->setPluginData( protocol(), accountId()+" displayName", groupName );
			m_groupList.replace( groupSn, fallBack );
		}
	}
	
	// Insert contacts
	{
		PHPArrayObject &contactlist = o.asMap()["FRIENDLIST"].asArray();
		
		// Cycle friendlist
		for( PHPArrayObject::iterator it = contactlist.begin(); it != contactlist.end(); ++it ) {
			PHPMapObject &m = it->asMap();
			
			Kopete::OnlineStatus ostat = SayclubProtocol::protocol()->sayclubOffline;
			if( !m["onlinestatus"].isNull() ) {
				string os = m["onlinestatus"].asString();
				if( os == "2" || os == "1" ) // Online
					ostat = SayclubProtocol::protocol()->sayclubOnline;
				else if( os == "3" || os == "5" ) // Away
					ostat = SayclubProtocol::protocol()->sayclubAway;
				else if( os == "4" ) // Busy
					ostat = SayclubProtocol::protocol()->sayclubBusy;
				else // Unknown
					ostat = SayclubProtocol::protocol()->sayclubOffline;
			}
			QString usrid = std2qstr( m["usrid"].asString() );
			QString username = std2qstrenc( m["dusername"].asString(), m_krcodec );
			QString msrl = QString::number( m["msrl"].asInt() );
			QString mynoti = std2qstrenc( m["mynoti"].asString(), m_krcodec );
			QString mydesc = std2qstrenc( m["desc"].asString(), m_krcodec );
			bool reversed = m["eachotherfriend"].asString() == "Y" ? true:false;
			
			SayclubContact *c = static_cast<SayclubContact *>( contacts()[ usrid ] );
			if( c ) { // if this contact is already in local contactlist
				if( !c->metaContact() ) {
					Kopete::MetaContact *metaContact = new Kopete::MetaContact();
					c->setMetaContact( metaContact );
					Kopete::ContactList::self()->addMetaContact( metaContact );
				}
				c->setOnlineStatus( ostat );
				c->setProperty( Kopete::Global::Properties::self()->nickName(), username );
				c->setProperty( SayclubProtocol::protocol()->propMsrl, msrl );
				c->setProperty( SayclubProtocol::protocol()->propMyDesc, mydesc );
				c->setProperty( SayclubProtocol::protocol()->propPersonalMessage, mynoti );
				c->setProperty( SayclubProtocol::protocol()->propReversed, reversed );
			}
			else { // not in local contactlist
				Kopete::MetaContact *metaContact = new Kopete::MetaContact();
				
				c = new SayclubContact( this, usrid, metaContact );
				c->setDeleted( true );
				c->setOnlineStatus( ostat );
				c->setProperty( Kopete::Global::Properties::self()->nickName(), username );
				c->setProperty( SayclubProtocol::protocol()->propMsrl, msrl );
				c->setProperty( SayclubProtocol::protocol()->propMyDesc, mydesc );
				c->setProperty( SayclubProtocol::protocol()->propPersonalMessage, mynoti );
				c->setProperty( SayclubProtocol::protocol()->propReversed, reversed );
				
				// Find current contact's occurance on the grouplist
				PHPArrayObject &groups = o.asMap()["GROUPLIST"].asArray();
				for( PHPArrayObject::iterator it2 = groups.begin(); it2 != groups.end(); ++it2 ) {
					if( it2->asMap()["flist"].isArray() ) continue; // When this map is empty, group is assumed to an array
					PHPMapObject &contactsingroup = it2->asMap()["flist"].asMap();
					for( PHPMapObject::iterator it3 = contactsingroup.begin(); it3 != contactsingroup.end(); ++it3 )
						if( std2qstr( it3->second.asString() ) == msrl ) {
							metaContact->addToGroup( m_groupList[ QString::number( it2->asMap()["groupsn"].asInt() ) ] );
							break;
						}
				}
				Kopete::ContactList::self()->addMetaContact( metaContact );
				c->setDeleted( false );
			}
		}
	}
	
	// Set myself
	{
		PHPMapObject &m = o.asMap()["MYPROFILE"].asMap();
		myself()->setProperty( Kopete::Global::Properties::self()->nickName(), std2qstrenc( m["dusername"].asString(), m_krcodec ) );
		myself()->setProperty( SayclubProtocol::protocol()->propMsrl, std2qstr( m["msrl"].asString() ) );
		myself()->setProperty( SayclubProtocol::protocol()->propMyDesc, QString( "" ) );
		myself()->setProperty( SayclubProtocol::protocol()->propPersonalMessage, std2qstrenc( m["mynoti"].asString(), m_krcodec ) );
	}
	
	A_REMOVEREQ( rq );
}

void SayclubAccount::slotSyncAddGroup( Kopete::Group *g )
{
	PHPObject o = PHPMapObject();
	PHPMapObject &m = o.asMap();
	
	m["method"] = string( "request_addfriend" );
	m["groupname"] = qstr2stdenc( g->displayName(), m_krcodec );
	m_server->sendCmd( "request_addfriend", o, this, SLOT( slotSyncAddGroupResponse( int ) ) );
}

void SayclubAccount::slotSyncAddGroupResponse( int rq )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	if( getErrorCode( o ) != -1 ) {
		emit errorMessage( SayclubServer::ErrorNormal, std2qstrenc( o.asMap()["errstr"].asString(), m_krcodec ) );
		return;
	}
	
	Kopete::Group *grp = NULL;
	QString groupSn = std2qstr( o.asMap()["groupsn"].asString() );
	QString groupName = std2qstrenc( o.asMap()["groupname"].asString(), m_krcodec );
	QPtrList<Kopete::Group> groupList = Kopete::ContactList::self()->groups();
	for( Kopete::Group *g = groupList.first(); g; g = groupList.next() ) {
		if( g->displayName() == groupName ) {
			grp = g;
			break;
		}
	}
	if( !grp ) {
		kdDebug( 14210 ) << k_funcinfo << "No group found!" << endl;
		return;
	}
	grp->setPluginData( protocol(), accountId()+" groupsn", groupSn );
	grp->setPluginData( protocol(), accountId()+" displayName", groupName );
	m_groupList.replace( groupSn, grp );
	
	A_REMOVEREQ( rq );
}

void SayclubAccount::slotRequestAddFriendResponse( int rq )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	if( getErrorCode( o ) != -1 ) {
		emit errorMessage( SayclubServer::ErrorNormal, std2qstrenc( o.asMap()["errstr"].asString(), m_krcodec ) );
		A_REMOVEREQ( rq );
		return;
	}
	
	cout << o << endl;
	A_REMOVEREQ( rq );
}

void SayclubAccount::slotSendMessage( int rq )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	if( o.asMap()["arrerrcode"].asArray().size() == 0 )
		server()->m_contactList[(long)rq]->manager( Kopete::Contact::CanCreate )->appendMessage( *server()->m_messageList[(long)rq] );
	else {
		Kopete::Message errMsg( server()->m_contactList[(long)rq], myself(), std2qstr( o.asMap()["arrerrstr"].asArray()[0].asString() ), Kopete::Message::Internal );
		server()->m_contactList[(long)rq]->manager( Kopete::Contact::CanCreate )->appendMessage( errMsg );
	}
	
	if( server()->m_contactList[(long)rq]->onlineStatus() == SayclubProtocol::protocol()->sayclubOffline ) {
		Kopete::Message offlineMsg( server()->m_contactList[(long)rq], myself(), QString::fromUtf8( "현재 이 아이디는 오프라인 상태이므로 당장은 쪽지를 받을 수 없습니다." ), Kopete::Message::Internal );
		server()->m_contactList[(long)rq]->manager( Kopete::Contact::CanCreate )->appendMessage( offlineMsg );
	}
	
	server()->m_contactList[(long)rq]->manager( Kopete::Contact::CanCreate )->messageSucceeded();
	
	A_REMOVEREQ( rq );
	server()->m_contactList.remove( rq );
	server()->m_messageList.remove( rq );
}

void SayclubAccount::slotChangeDisplayName()
{
	if( !m_connected ) {
		emit errorMessage( SayclubServer::ErrorInformation, QString::fromUtf8( "먼저 로그인하세요." ) );
		return;
	}
	
	bool ok;
	QString name = KInputDialog::getText( QString::fromUtf8( "대화명 변경 - 세이클럽 플러그인" ), QString::fromUtf8( "변경하실 대화명을 입력하세요. 최대 12바이트까지 입력하실 수 있습니다." ), myself()->property( Kopete::Global::Properties::self()->nickName() ).value().toString(), &ok );
	
	if( ok ) {
		QCString encNick = m_krcodec->fromUnicode( name );
		if( encNick.length() > 12 ) {
			emit errorMessage( SayclubServer::ErrorNormal, QString::fromUtf8( "닉네임은 최대 12바이트까지 입력하실 수 있습니다." ) );
			return;
		}
		
		PHPObject o = PHPMapObject();
		PHPMapObject &m = o.asMap();
		m["method"] = string( "request_updateuserinfo" );
		m["MEMBER_NICKNAME"] = qcstr2std( encNick );
		m_server->sendCmd( "request_updateuserinfo", o );
		
		myself()->setProperty( Kopete::Global::Properties::self()->nickName(), name );
	}
}

void SayclubAccount::slotDisposeRequest( int rq )
{
	PHPObject &o = *m_server->m_objectList[(long)rq];
	
	if( getErrorCode( o ) != -1 ) {
		emit errorMessage( SayclubServer::ErrorNormal, std2qstrenc( o.asMap()["errstr"].asString(), m_krcodec ) );
		A_REMOVEREQ( rq );
		return;
	}
	
	A_REMOVEREQ( rq );
}

void SayclubAccount::updateContactStatus()
{
	for( QDictIterator<Kopete::Contact> itr( contacts() ); itr.current(); ++itr )
		itr.current()->setOnlineStatus( myself()->onlineStatus() );
}

int SayclubAccount::getErrorCode( const PHPObject &o )
{
	if( !HAS_KEY( o.asMap(), "errcode" ) )
		return -1;
	
	return o.asMap()["errcode"].asInt();
}

// Helper function; Find a contact by Msrl(a kind of unique identifier) value
SayclubContact* SayclubAccount::getContactByMsrl( QString msrl )
{
	for( QDictIterator<Kopete::Contact> it( contacts() ); it.current(); ++it )
		if( it.current()->property( "msrl" ).value() == msrl )
			return static_cast<SayclubContact *>(it.current());
	
	return NULL;
}

void SayclubAccount::requestAddFriend( QString id, QString desc, QString group )
{
	PHPObject o = PHPMapObject();
	PHPMapObject &m = o.asMap();

	m["method"] = string( "request_addfriend" );
	m["targetmsrl"] = string( "" );
	m["targetid"] = qstr2std( id );
	m["memo"] = qstr2stdenc( desc, m_krcodec );
	m["groupname"] = qstr2stdenc( group, m_krcodec );

	m_server->sendCmd( "request_addfriend", o, this, SLOT( slotRequestAddFriendResponse( int ) ) );
}

// Helper function; Convert Onlinestatus to Id
string SayclubAccount::statusToId( Kopete::OnlineStatus &status )
{
	kdDebug ( 14210 ) << k_funcinfo << endl;
	
	if( status == SayclubProtocol::protocol()->sayclubOnline )
		return string( "2" ); // Online
	else if( status == SayclubProtocol::protocol()->sayclubAway )
		return string( "3" ); // Away
	else if( status == SayclubProtocol::protocol()->sayclubBusy )
		return string( "4" ); // Busy
	else if( status == SayclubProtocol::protocol()->sayclubOffline )
		return string( "0" ); // Offline
	else
		return string(); // And no
}

#include "sayclubaccount.moc"
