<?php
/* vim: set expandtab tabstop=4 shiftwidth=4: */
// +--------------------------------------------------------+
// | PHP version 5.x                                        |
// +--------------------------------------------------------+
// | Copyright : Song Hyo-Jin <shj at xenosi.de>            |
// +--------------------------------------------------------+
// | License : BSD                                          |
// +--------------------------------------------------------+
//
// $Id: PDOExt.inc.php, 2009. 3. 23. crucify Exp $

class SHJPDOExt
{
	public $query = false, $table = '';
	protected $class = null, $transaction = false, $charset = false, $dsn = '';
	private $stmt = 'SHJPDOStatement', $msel = 'SHJPDOSelect', $mins = 'SHJPDOInsert';

	public function __construct($conninfo, $user = '', $pass = '', $pconnect = false)
	{
		if($user == '') {
			$user = null;
		}
		if($pass == '') {
			$pass = null;
		}
		$options = array();
		if($pconnect) {
			$options[PDO::ATTR_PERSISTENT] = true;
		}
		$this->class = new PDO($conninfo, $user, $pass, $options);

		if(!$this->class) {
			throw new Exception('connect error.');
		}
		$this->preset();
	}
	
	public function getDsn()
	{
		return $this->dsn;
	}
	
	protected function setDsn($dsn)
	{
		$this->dsn = $dsn;
	}
	
	protected function setStmt($stmt)
	{
		$this->stmt = 'SHJPDO'.$stmt.'Statement';
	}
	
	protected function setSelect($msel)
	{
		$this->msel = 'SHJPDO'.$msel.'Select';
	}
	
	protected function setInsert($mins)
	{
		$this->mins = 'SHJPDO'.$mins.'Insert';
	}
	
	protected function preset()
	{
		$this->class->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		if(false !== $this->charset) {
			$this->exec('SET NAMES \''.$this->charset.'\'');
		}
	}

	public function __destruct()
	{
		if($this->transaction) {
			$this->rollback();
		}
	}

	public function begin()
	{
		if($this->transaction) {
			throw new Exception('already begined.');
		}
		$this->class->beginTransaction();
		$this->transaction = true;
	}

	public function commit()
	{
		if(!$this->transaction) {
			throw new Exception('not begined.');
		}
		$this->class->commit();
		$this->transaction = false;
	}

	public function rollback()
	{
		if(!$this->transaction) {
			throw new Exception('not begined.');
		}
		$this->class->rollBack();
		$this->transaction = false;
	}

	public function exec($query = false)
	{
		if(false !== $query) {
			$this->query = $query;
		}
		return $this->class->exec($this->query);
	}

	public function query($query = false)
	{
		if(false !== $query) {
			$this->query = $query;
		}

		$stmt = $this->class->query($this->query);
		return new $this->stmt($stmt);
	}

	public function prepare($query = false)
	{
		if(false !== $query) {
			$this->query = $query;
		}
		$stmt = $this->class->prepare($this->query);
		return new $this->stmt($stmt);
	}
	
	public function id()
	{
		return $this->class->lastInsertId();
	}
	
	public function makeQuery($type)
	{
		switch($type) {
			case 'select':
				return new $this->msel($this);
			case 'insert':
			case 'update':
				return new $this->mins($this);
			default:
				return false;
		}
	}

	public function readLobs(&$rows, $callback = '')
	{
		if(is_resource($rows)) {
			$res = '';
			while(!feof($rows)) {
				$res .= fgets($rows, 4096);
			}
			$rows = $res;
			if($callback != '') {
				$rows = $callback($rows);
			}
			return;
		}
		if(!is_array($rows) && !is_object($rows)) return;
		foreach($rows as &$row) {
			$this->readLobs($row, $callback);
		}
	}
}

function escape_like($query)
{
	$pattern[] = '/_/';
	$replacement[] = '\_';

	$pattern[] = '/%/';
	$replacement[] = '\%';

	return preg_replace($pattern, $replacement, $query);
}

class SHJPDOStatement
{
	public $class;

	public function __construct($stmt)
	{
		$this->class = $stmt;
	}

	public function __destruct()
	{
		$this->class = null;
	}
	
	public function bind(&$fields)
	{
		foreach($fields as $key => &$value) {
			if(is_array($value)) {
				$this->class->bindParam($key, $value[0], PDO::PARAM_LOB);
			} else if(is_resource($value)) {
				$this->class->bindParam($key, $value, PDO::PARAM_LOB);
			} else {
				$this->class->bindParam($key, $value);
			}
		}
		return true;
	}

	public function field()
	{
		return $this->class->fetchColumn();
	}

	public function fetch()
	{
		return $this->class->fetch(PDO::FETCH_OBJ);
	}

	public function fetch_all()
	{
		$ret = $this->class->fetchAll(PDO::FETCH_OBJ);
		if(count($ret) == 0) {
			return false;
		}
		return $ret;
	}

	public function fetch_assoc()
	{
		return $this->class->fetch(PDO::FETCH_ASSOC);
	}

	public function fetch_assoc_all()
	{
		$ret = $this->class->fetchAll(PDO::FETCH_ASSOC);
		if(count($ret) == 0) {
			return false;
		}
		return $ret;
	}

	public function exec($fields = false)
	{
		if(false === $fields) {
			$this->class->execute();
		} else {
			foreach($fields as $key => $value) {
				$field[':'.$key] = $value;
			}
			$this->class->execute($field);
		}
		return $this;
	}

	public function rows()
	{
		return $this->class->rowCount();
	}
}

class SHJPDOSelect
{
	public $tables, $fields, $wfields, $wheres, $orders, $limit, $offset;
	protected $class;
	
	public function __construct(&$class)
	{
		$this->class = &$class;
		$this->tables = array();
		$this->fields = array();
		$this->wfields = null;
		$this->wheres = array();
		$this->orders = array();
		$this->limit = false;
		$this->offset = false;
	}
	
	public function compile()
	{
		$query = 'SELECT ';
		if(count($this->fields) == 0) {
			$query .= '*';
		} else {
			$query .= implode(', ', $this->fields);
		}
		$query .= ' FROM '.implode(', ', $this->tables);
		if(count($this->wheres) != 0) {
			$query .= ' WHERE ('.implode(') AND (', $this->wheres).')';
		}
		if(count($this->orders) != 0) {
			$query .= ' ORDER BY '.implode(', ', $this->orders);
		}
		if($this->limit) {
			$query .= ' LIMIT :limit';
		}
		if($this->offset) {
			$query .= ' OFFSET :offset';
		}
		return $query;
	}
	
	public function prepare()
	{
		$fields = null;
		if($this->limit) {
			$fields->limit = &$this->limit;
		}
		if($this->offset) {
			$fields->offset = &$this->offset;
		}
		$query = $this->compile();
		$stmt = $this->class->prepare($query);
		if($fields != null) {
			$stmt->bind($fields);
		}
		if($this->wfields != null) {
			$stmt->bind($this->wfields);
		}
		return $stmt;
	}
}

class SHJPDOInsert
{
	public $table, $wheres, $fields, $wfields;
	protected $class;
	
	public function __construct(&$class)
	{
		$this->class = &$class;
		$this->table = '';
		$this->wheres = array();
		$this->fields = null;
		$this->wfields = null;
	}
	
	public function insertCompile()
	{
		if($this->fields == null) {
			throw new Exception('SHJPDOExtInsert::fields is null');
		}
		$this->class->table = $this->table;
		foreach($this->fields as $key => &$value) {
			$keys[] = $key;
			$values[] = ':'.$key;
		}
		return 'INSERT INTO '.$this->table.' ('.implode(', ', $keys).') VALUES ('.implode(', ', $values).')';
	}
	
	public function insert()
	{
		$query = $this->insertCompile();
		$stmt = $this->class->prepare($query);
		$stmt->bind($this->fields);
		return $stmt;
	}
	
	public function updateCompile()
	{
		if($this->fields == null) {
			throw new Exception('SHJPDOExtInsert::fields is null');
		}
		if($this->wfields == null) {
			throw new Exception('SHJPDOExtInsert::wfields is null');
		}
		if(count($this->wheres) == 0) {
			throw new Exception('SHJPDOExtInsert::wheres is null');
		}
		$this->class->table = $this->table;
		foreach($this->fields as $key => &$value) {
			$keys[] = $key.' = :'.$key;
		}
		return 'UPDATE '.$this->table.' SET '.implode(', ', $keys).' WHERE ('.implode(') AND (', $this->wheres).')';
	}
	
	public function update()
	{
		$query = $this->updateCompile();
		$stmt = $this->class->prepare($query);
		$stmt->bind($this->fields);
		$stmt->bind($this->wfields);
		return $stmt;
	}
}
