<?php
/* vim: set expandtab tabstop=4 shiftwidth=4: */
// +--------------------------------------------------------+
// | PHP version 5.x                                        |
// +--------------------------------------------------------+
// | Copyright : Song Hyo-Jin <shj at xenosi.de>            |
// +--------------------------------------------------------+
// | License : BSD                                          |
// +--------------------------------------------------------+
//
// $Id: PDOPostgreSQL.inc.php, 2009. 3. 24. crucify Exp $

class SHJPDOORACLE
{
	public $query, $table, $transaction_type, $conn;
	protected $transaction;
	
	public function __construct($conninfo, $pconnect = false)
	{
		$this->query = false;
		$this->transaction = false;
		$this->transaction_type = OCI_COMMIT_ON_SUCCESS;
		$conndata = parse_ini_file($conninfo);
		putenv('NLS_LANG=KOREAN_KOREA.KO16MSWIN949');
		$conntext = '';
		if(isset($conndata['host'])) {
			$conntext = '(DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)';
			$conntext .= '(HOST=' . $conndata['host'] . ')';
			$conntext .= '(PORT=' . (isset($conndata['port']) ? $conndata['port'] : '1521') . '))';
			if(isset($conndata['sid'])) {
				$conntext .= '(CONNECT_DATA=(SERVICE_NAME=' . $conndata['sid'] . '))';
			}
			$conntext .= ')';
		}
		$charset = '';
		if(isset($conndata['charset'])) {
			$charset = $conndata['charset'];
		}
		if($pconnect) {
			$this->conn = oci_pconnect($conndata['user'], $conndata['pass'], $conntext, $charset);
		} else {
			$this->conn = oci_new_connect($conndata['user'], $conndata['pass'], $conntext, $charset);
		}
	}
	
	public function getDsn()
	{
		return 'oracle';
	}
	
	public function __destruct()
	{
		if($this->transaction) {
			$this->rollback();
		}
	}
	
	public function begin()
	{
		if($this->transaction) {
			throw new Exception('already begined.');
		}
		$this->transaction_type = OCI_DEFAULT;
		$this->transaction = true;
	}
	
	public function commit()
	{
		if(! $this->transaction) {
			throw new Exception('not begined.');
		}
		oci_commit($this->conn);
		$this->transaction = false;
		$this->transaction_type = OCI_COMMIT_ON_SUCCESS;
	}
	
	public function rollback()
	{
		if(! $this->transaction) {
			throw new Exception('not begined.');
		}
		oci_rollback($this->conn);
		$this->transaction = false;
		$this->transaction_type = OCI_DEFAULT;
	}
	
	public function exec($query = false)
	{
		if(false !== $query) {
			$this->query = $query;
		}
		return oci_execute(oci_parse($this->conn, $this->query), $this->transaction_type);
	}
	
	public function escape_string($str)
	{
		return pg_escape_string($str);
	}
	
	public function query($query = false)
	{
		$stmt = $this->prepare($query);
		$stmt->exec();
		return $stmt;
	}
	
	public function prepare($query = false)
	{
		if(false !== $query) {
			$this->query = $query;
		}
		return new SHJPDOORACLEStatement(oci_parse($this->conn, $this->query), $this);
	}
	
	public function id($seq_name, $seq = 0)
	{
		if($seq > 0) {
			throw new Exception('SHJPDOORACLE::id() setval not supported.');
		} else if($seq == -1) {
			return $this->query('SELECT '.$seq_name.'.NEXTVAL FROM DUAL')->field();
		} else {
			return $this->query('SELECT '.$seq_name.'.CURRVAL FROM DUAL')->field();
		}
	}
	public function makeQuery($type)
	{
		switch($type) {
			case 'select':
				return new SHJPDOORACLESelect($this);
			case 'insert':
			case 'update':
				return new SHJPDOORACLEInsert($this);
			default:
				return false;
		}
	}
	
	public function readLobs(&$rows, $callback = '')
	{
		if(is_object($rows) && method_exists($rows, 'read')) {
			$res = $rows->read($rows->size());
			oci_free_descriptor($rows);
			$rows = $res;
			if($callback != '') {
				$rows = $callback($rows);
			}
			return;
		}
		if(!is_array($rows) && !is_object($rows)) return;
		foreach($rows as &$row) {
			$this->readLobs($row, $callback);
		}
	}
}

class SHJPDOORACLEStatement
{
	protected $stmt, $class, $lobs = array();
	
	public function __construct(&$stmt, &$class)
	{
		$this->stmt = &$stmt;
		$this->class = &$class;
	}
	
	public function __destruct()
	{
		if(count($this->lobs) != 0) {
			foreach($this->lobs as &$lob) {
				oci_free_descriptor($lob['p']);
			}
		}
		oci_free_statement($this->stmt);
	}
	
	public function bind(&$fields)
	{
		foreach($fields as $key => &$value) {
			if(is_array($value)) {
				$this->lobs[$key]['p'] = oci_new_descriptor($this->class->conn, OCI_DTYPE_LOB);
				$this->lobs[$key]['v'] = &$value[0];
				if(count($value) == 1) {
					oci_bind_by_name($this->stmt, ':'.$key, $this->lobs[$key]['p'], -1, SQLT_BLOB);
				} else {
					oci_bind_by_name($this->stmt, ':'.$key, $this->lobs[$key]['p'], -1, SQLT_CLOB);
				}
				if(is_resource($value[0])) {
					$this->lobs[$key]['t'] = 'fp';
				} else {
					$this->lobs[$key]['t'] = 'string';
				}
			} else if(is_resource($value)) {
				$this->lobs[$key]['p'] = oci_new_descriptor($this->class->conn, OCI_DTYPE_LOB);
				$this->lobs[$key]['v'] = &$value;
				oci_bind_by_name($this->stmt, ':'.$key, $this->lobs[$key]['p'], -1, SQLT_BLOB);
				$this->lobs[$key]['t'] = 'fp';
			} else {
				oci_bind_by_name($this->stmt, ':'.$key, $value);
			}
		}
		return true;
	}
	
	public function field()
	{
		if(false === ($fields = oci_fetch_array($this->stmt, OCI_NUM | OCI_RETURN_NULLS))) {
			return false;
		}
		return $fields[0];
	}
	
	public function fetch()
	{
		if(false === ($fields = oci_fetch_array($this->stmt, OCI_ASSOC | OCI_RETURN_NULLS))) {
			return false;
		}
		$ret = null;
		foreach($fields as $key => $value) {
			$ret->{strtolower($key)} = $value;
		}
		unset($ret->rnum);
		return $ret;
	}
	
	public function fetch_all()
	{
		while(false !== ($field = $this->fetch())) {
			$fields[] = $field;
		}
		if(!isset($fields)) {
			return false;
		}
		return $fields;
	}
	
	public function fetch_assoc()
	{
		if(false === ($fields = oci_fetch_array($this->stmt, OCI_ASSOC | OCI_RETURN_NULLS))) {
			return false;
		}
		foreach($fields as $key => $value) {
			$ret[strtolower($key)] = $value;
		}
		unset($ret['rnum']);
		return $ret;
	}
	
	public function fetch_assoc_all()
	{
		while(false !== ($field = $this->fetch_assoc())) {
			$fields[] = $field;
		}
		if(! isset($fields)) {
			return false;
		}
		return $fields;
	}
	
	public function exec($fields = false)
	{
		if(false !== $fields) {
			$this->bind($fields);
		}
		oci_execute($this->stmt, $this->class->transaction_type);
		if(array($this->lobs) != 0) {
			foreach($this->lobs as &$lob) {
				switch($lob['t']) {
					case 'fp':
						while(!feof($lob['v'])) {
							$lob['p']->write(fgets($lob['v'], 4096));
						}
						break;
					default:
						$lob['p']->write($lob['v']);
						break;
				}
			}
		}
		return $this;
	}
	
	public function rows()
	{
		return oci_num_rows($this->stmt);
	}
}

class SHJPDOORACLESelect
{
	public $tables, $fields, $wfields, $wheres, $orders, $limit, $offset;
	protected $class;
	
	public function __construct(&$class)
	{
		$this->class = &$class;
		$this->tables = array();
		$this->fields = array();
		$this->wfields = null;
		$this->wheres = array();
		$this->orders = array();
		$this->limit = false;
		$this->offset = false;
	}
	
	public function compile($lock = false)
	{
		if(count($this->fields) == 0) {
			$this->fields[] = '*';
		}
		$query = 'SELECT xenoq1.*, ROWNUM AS RNUM FROM (SELECT '.implode(', ', $this->fields).' FROM '.implode(', ', $this->tables);
		if(count($this->wheres) != 0) {
			$query .= ' WHERE (' . implode(') AND (', $this->wheres) . ')';
		}
		if(count($this->orders) != 0) {
			$query .= ' ORDER BY ' . implode(', ', $this->orders);
		}
		if($lock) {
			$query .= ' FOR UPDATE';
		}
		$query .= ') xenoq1';
		if($this->limit) {
			$query .= ' WHERE '.(($this->offset) ? 'ROWNUM <= :offset + :limit' : 'ROWNUM <= :limit');
		}
		if($this->offset) {
			$query = 'SELECT * FROM ('.$query.') xenoq2 WHERE RNUM > :offset';
		}
		return $query;
	}
	
	public function prepare($lock = false)
	{
		$fields = null;
		if($this->limit) {
			$fields->limit = &$this->limit;
		}
		if($this->offset) {
			$fields->offset = &$this->offset;
		}
		$stmt = $this->class->prepare($this->compile($lock));
		if($fields != null) {
			$stmt->bind($fields);
		}
		if($this->wfields != null) {
			$stmt->bind($this->wfields);
		}
		return $stmt;
	}
}

class SHJPDOORACLEInsert
{
	public $table, $wheres, $fields, $wfields;
	protected $class;
	
	public function __construct(&$class)
	{
		$this->class = &$class;
		$this->table = '';
		$this->wheres = array();
		$this->fields = null;
		$this->wfields = null;
	}

	public function insertCompile()
	{
		if($this->fields == null) {
			throw new Exception('SHJPDOORACLEInsert::fields is null');
		}
		$this->class->table = $this->table;
		foreach($this->fields as $key => &$value) {
			if(is_array($value)) {
				$keys[] = $key;
				if(count($value) == 1) {
					$values[] = 'EMPTY_BLOB()';
				} else {
					$values[] = 'EMPTY_CLOB()';
				}
				$returns[] = $key;
				$returnvalues[] = ':'.$key;
			} else if(is_resource($value)) {
				$keys[] = $key;
				$values[] = 'EMPTY_BLOB()';
				$returns[] = $key;
				$returnvalues[] = ':'.$key;
			} else {
				$keys[] = $key;
				$values[] = ':'.$key;
			}
		}
		$query = 'INSERT INTO '.$this->table.' ('.implode(', ', $keys).') VALUES ('.implode(', ', $values).')';
		if(isset($returns)) {
			$query .= ' RETURNING '.implode(', ', $returns).' INTO '.implode(', ', $returnvalues);
		}
		return $query;
	}
	
	public function insert()
	{
		$stmt = $this->class->prepare($this->insertCompile());
		$stmt->bind($this->fields);
		return $stmt;
	}
	
	public function updateCompile($result = false)
	{
		if($this->fields == null) {
			throw new Exception('SHJPDOORACLEInsert::fields is null');
		}
		if($this->wfields == null) {
			throw new Exception('SHJPDOORACLEInsert::wfields is null');
		}
		if(count($this->wheres) == 0) {
			throw new Exception('SHJPDOORACLEInsert::wheres is null');
		}
		$this->class->table = $this->table;
		foreach($this->fields as $key => &$value) {
			if(is_array($value)) {
				if(count($value) == 1) {
					$keys[] = $key.' = EMPTY_BLOB()';
				} else {
					$keys[] = $key.' = EMPTY_CLOB()';
				}
				$returns[] = $key;
				$returnvalues[] = ':'.$key;
			} else if(is_resource($value)) {
				$keys[] = $key.' = EMPTY_BLOB()';
				$returns[] = $key;
				$returnvalues[] = ':'.$key;
			} else {
				$keys[] = $key.' = :'.$key;
			}
		}
		$query = 'UPDATE '.$this->table.' SET '.implode(', ', $keys).' WHERE ('.implode(') AND (', $this->wheres).')';
		if(isset($returns)) {
			$query .= ' RETURNING '.implode(', ', $returns).' INTO '.implode(', ', $returnvalues);
		}
		return $query;
	}

	public function update($result = false)
	{
		$stmt = $this->class->prepare($this->updateCompile($result));
		$stmt->bind($this->fields);
		$stmt->bind($this->wfields);
		if($this->returns != null) {
			$stmt->bind($this->returns);
		}
		return $stmt;
	}	
}